<?php
/**
 * Tiny.cc REST API v2 client
 * 
 * This client works with tiny.cc URL shortening services using API v2.0.3.
 * 
 * Methods prefixed with 'mass_' use curl_multi to establish several parallel connections for faster processing.
 * On communication error these method DOESN'T throw exceptions. Instead they returne error codes for failed URLs.
 * But they still may throw exceptions if invalid arguments passed.
 * 
 * @copyright	2016 Tiny.cc
 * @author		Alexey Gorshunov <ag@blazing.pro>
 * @license		MIT
 * @package		tinycc/classic_client
 */
 
namespace tinycc;

require_once "classic_client_exception.php";

/**
 * Client class
 */
class classic_client
{
	protected $api_root_url;
	protected $username;
	protected $api_key;
	protected $api_version = "2.0.3";
	protected $unknown_error = array("code"=>101, "message"=>'Unknown error','details'=>'');
	
	protected $selection = null;
	
/**
 * Client contructor
 * @param	array	$config		Required keys: 'api_root_url', 'username', 'api_key', Optional keys: 'batch_operations_limit', 'parallel_streams'
 * @api
 */	
	public function __construct(array $config)
	{
		$this->api_root_url = @$config['api_root_url'];
		$this->username = @$config['username'];
		$this->api_key = @$config['api_key'];
		$this->batch_operations_limit = isset($config['batch_operations_limit']) ? $config['batch_operations_limit'] : 30;
		$this->parallel_streams = isset($config['parallel_streams']) ? $config['parallel_streams'] : 4;
		
		if(empty($this->api_root_url)
			OR filter_var($this->api_root_url, FILTER_VALIDATE_URL) === false){
				throw classic_client_exception::invalid_argument(__METHOD__,"api_root_url");
		}
		
		$this->api_root_url = rtrim($this->api_root_url,"/")."/";
		
		if(empty($this->username)){
			throw classic_client_exception::invalid_argument(__METHOD__,"username");
		}

		if(empty($this->api_key)){
			throw classic_client_exception::invalid_argument(__METHOD__,"api_key");
		}
	}
	
/**
 * Shorten many long URLs at once
 * @api
 * @param	array 	$long_urls each element of array is a string representing long URL
 * @param	array	$data	optional URL properties. Possible keys: 'custom_hash', 'note', 'is_qrcode', 'favorite', 'email_stats', 'expiration_date'
 * @return	array	shortened URLs properties
 */	
	public function mass_shorten(array $long_urls, array $data=array())
	{
		if(empty($long_urls)){
			throw classic_client_exception::invalid_argument(__METHOD__,"long_urls");
		}

		if(!is_array($data)){
			throw classic_client_exception::invalid_argument(__METHOD__,"data");
		}

		$results = $this->simple_api_call('POST','shorten_batch',array(),array(
			'longUrls'=>$long_urls
		));
		return $results;
	}
	

/**
 * @internal
 */	
	protected function prepare_curl($method, $resource, $get_params = array(), $post_data = null)
	{
		$curl = curl_init(); //cUrl Init

		$get_params['login'] = $this->username;
		$get_params['apiKey'] = $this->api_key;
		$get_params['version'] = $this->api_version;
		$get_params['format'] = 'json';
		$get_params['c'] = 'rest_api';
		$get_params['m'] = $resource;

		if($post_data){
			curl_setopt($curl, CURLOPT_POST, 1); 
			curl_setopt($curl, CURLOPT_POSTFIELDS, http_build_query($post_data));
			//$get_params = array_merge($get_params, $post_data);
		}
		
		$query_string = empty($get_params) ? "" : ("?".http_build_query($get_params));
		
		curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
		curl_setopt($curl, CURLOPT_HEADER, 0); // show headers
		curl_setopt($curl, CURLOPT_FOLLOWLOCATION, 1);
		curl_setopt($curl, CURLOPT_HTTPHEADER, array(
			'Accept: application/json',
			//'Content-type: application/x-www-form-urlencoded',
			"Cache-control: no-cache",
		));
		curl_setopt($curl, CURLOPT_URL, $this->api_root_url.$query_string);
		//print $this->api_root_url.$query_string; exit(0);
		curl_setopt($curl, CURLOPT_CUSTOMREQUEST, strtoupper($method));
		
		return $curl;
	}
	
/**
 * @internal
 */	
	protected function simple_api_call($method, $resource, $get_params = array(), $post_data = null)
	{
		$curl = $this->prepare_curl($method, $resource, $get_params, $post_data);
		return $this->curl_request($curl);
	}
	
/**
 * @internal
 */	
	protected function curl_request($curl)
	{
		$start_time = microtime(true);
		
		$response = curl_exec($curl);
		$response_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		
		if (!$response) {
			throw classic_client_exception::curl_error($curl);
		}
		$data = json_decode($response, true);
		curl_close($curl);
		
		if($response_code != '200'){
			throw classic_client_exception::api_error($response);
		}
		
		$data['meta'] = array('request_time' => (microtime(true) - $start_time));
		return $data;
	}
	
}


